
function [Dk, Dd1, Dd2, lambda] = getDecision(A0,B0,n_k)

% This function solves for an individual's decision rule, assuming that
% agent i's state/choice variables evolve as such:

% A0*E_i_t[z_t+1] = B0*z_t + delta_t where 
% z_t =[f_t k_t-1]'
% f_t   is a 1xn_f matrix of time t choice variables
% k_t-1 is a 1xn_k is time t state variables. Note that getDecision takes 
% n_k as input because it tells us where in z_t the choice variables end
% and the state variables begin.

% As output, we obtain 3 matrices which characterize the solution of the
% decision rule: (1) Dk (2) Dd1 (3) Dd2 (4) Lambda.
% See "General Formulation of Individual Decision Rules" for a
% derivation and a description of how these three matrices are used in
% writing the decision rule.

%% get dimensions
n_z=size(A0,1);
n_f=n_z-n_k;

A=A0\B0;

%% Compute matrix of left eigenvectors of A

% compute matrix of left eigenvectors two different ways, and select the
% method that yields the smallest errors
[LE1, MU1]=eig(A'); LE1=LE1';
[LE2, MU2]=eig(A','nobalance'); LE2=LE2'; MU2=MU2';


if sum(sum(abs(LE1*A-MU1*LE1)))>sum(sum(abs(LE2*A-MU2*LE2)))
    LE=LE2; MU=MU2;
else
    LE=LE1; MU=MU1;
end

%% Check for solvability
LUE=abs(diag(MU))>1.000001;
NUE=sum(LUE);
if NUE>0
    LE=LE(LUE,:); MU=MU(LUE,LUE);
    if NUE>rank(LE(:,1:NUE),10^-5)
       error('Rank condition violated');
    end
else
    LE=[0]; MU=[0];
end

if NUE>n_f
    error('Too many unstable eigenvectors');
elseif NUE<n_f
    error('Too few unstable eigenvectors');
end


%% Construct Output Matrices
V=LE';
lambda=MU;

if sum((abs(V'*A-lambda*V')))>0.00001;
   error('An error...');
end

% Extract Vf_prime Vk_prime
Vfp=LE(:,1:n_f); Vkp=LE(:,n_f+1:end);
if rank(Vfp)>size(Vfp,1)
   error('Vf_prime is singular');
end

Dk=-1*Vfp\Vkp;
Dd1=-1*Vfp\inv(lambda);
Dd2=LE*inv(A0);

%% Print output matrices

disp('Dk=');
disp(Dk);
disp('Dd1=');
disp(Dd1);
disp('Dd2=');
disp(Dd2);
disp('lambda=');
disp(lambda);




        
        